/******************************************************************************\

Module Name:    LVideoOut.c

Description:    LVideOut sample implementation file.

References:     LVideoOut.doc Revision 1.2.
                Matrox Liberatus specifications

    Copyright (c) 2014, Matrox Graphics Inc.
    All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/
#include "Liberatus.h"
#include "LBuffer.h"
#include "LVideoOut.h"

/********************************************************************\

Function:       main

Description:    LVideoOut sample entry point.

Parameters:     None.

Return Value:   0 on success, -1 otherwise.

Comments:       - This sample code creates buffers, enables the first
                  video output, flips between buffers and retrieves 
                  counters.

\********************************************************************/
int main()
{
    LDevice_Handle              hDevice             = MNULL;
    LBuffer_Handle              hBuffer1            = MNULL;
    LBuffer_Handle              hBuffer2            = MNULL;
    LVideoOut_Handle            hVideoOut           = MNULL;
    LVideo_VidParam             oVidParam           = {0};
    LBuffer_VideoAttributes     oVideoAttribute     = {0};
    MUINT32                     uiIndex             = 0;
    MUINT64                     uiFieldCounter      = 0;
    MUINT64                     uiSetBufferCounter  = 0;
    MUINT64                     uiTickRefCounter    = 0;
    MBOOL32                     bIsTopField         = MTRUE;

    Liberatus_Load();

    /* Get the handle of the primary device.                        */
    hDevice = Liberatus_GetDevice(0);

    /* Get the handle of the primary video output in exclusive mode */
    LVideoOut_GetHandle(hDevice,
                        0,
                        LAccessMode_READWRITE_EXCLUSIVE,
                        &hVideoOut);

    /* Retrieve the best video parameters for the monitor detected  */
    LVideoOut_GetOptimalVideoParameters(hVideoOut,
                                        LVideo_SignalType_USEDEFAULT,
                                        &oVidParam);

    /* Create 2 video buffers in RGB 32 bits                        */
    oVideoAttribute.eAttributeType  = LBuffer_Type_VIDEO;
    oVideoAttribute.ePixelFormat    = LPixelFormat_B8G8R8A8;
    oVideoAttribute.uiWidth         = oVidParam.uiHActive;
    oVideoAttribute.uiHeight        = oVidParam.uiVActive;

    LBuffer_Create(hDevice,
                   &oVideoAttribute.eAttributeType,
                   &hBuffer1);

    LBuffer_Create(hDevice,
                   &oVideoAttribute.eAttributeType,
                   &hBuffer2);

    /* ... Here draw something into the buffers  ...                */
    /* LBuffer_BeginAccess() and LBuffer_EndAccess() can be use.    */

    /* Enable the video output and optionally keep the sync disable.*/
    LVideoOut_Enable(hVideoOut,
                     LVideo_SignalType_USEDEFAULT,
                     &oVidParam,
                     hBuffer1,
                     LVideoOut_PowerState_OFF);

    /* Enable the video output sync.                                */
    LVideoOut_SetPowerState(hVideoOut,
                            LVideoOut_PowerState_ON);


    /* Flip between buffers for 100 iterations.                     */
    for (uiIndex = 0; uiIndex < 100; uiIndex++)
    {
        /* Retrieve field, flip and tick counters                   */
        LVideoOut_GetCounters(hVideoOut,
                              &uiFieldCounter,
                              &uiSetBufferCounter,
                              &uiTickRefCounter,
                              &bIsTopField);

        /* Do the field alternation.                                */
        /*  Note: the top field is don't care when progressive      */
        /*        timings are use (oVidParam.bInterlaced == MFALSE) */
        /*        Otherwise when have to set the top and the bottom */
        /*        fields in alternation .                           */
        bIsTopField = !bIsTopField;

        /* Flip buffer for the next field and wait until the        */
        /* previous buffer is not use anymore                       */
        LVideoOut_SetBuffer(hVideoOut,
                            hBuffer2,
                            bIsTopField,
                            LVideoOut_FlipType_NEXT_FIELD_WAIT);

        /* Do the field alternation again                           */
        bIsTopField = !bIsTopField;

        /* Flip buffer for the next field and wait until the        */
        /* previous buffer is not use anymore                       */
        LVideoOut_SetBuffer(hVideoOut,
                            hBuffer1,
                            bIsTopField,
                            LVideoOut_FlipType_NEXT_FIELD_WAIT);
    }

    /* Disable video output and free all resources                  */
    LVideoOut_Disable(hVideoOut);

    LVideoOut_ReleaseHandle(hVideoOut);

    LBuffer_Destroy(hBuffer1);

    LBuffer_Destroy(hBuffer2);

    Liberatus_UnLoad();

    return 0;
}

